#!/usr/bin/env python
'''
transmutation class for markers

It performs line-by-line substitutions based on markers embedded in comments.

Mark up source class with following comment markers:
// DATA_TYPE      - abstract dataset constant
// CLASS_TYPE     - boxed primitive class
// PRIM_TYPE      - java primitive type
// PRIM_TYPE_LONG - java primitive type (cast to long first if integer)
// GET_ELEMENT    - use get element method
// FROM_OBJECT    - use convert from object method
// REAL_ONLY      - keep line when a real dataset
// BOOLEAN_OMIT   - omit line when boolean dataset
// BOOLEAN_USE    - use commented out code
// BOOLEAN_FALSE  - return false when boolean dataset
// BOOLEAN_ZERO   - return zero when boolean dataset
// FORMAT_STRING  - format string for getString method
// DEFAULT_VAL    - default value for expanded dataset
// INT_EXCEPTION  - surround with try/catch for integer arithmetic exception
// INT_ZEROTEST   - use commented out code for testing for integer zero
// OMIT_SAME_CAST - omit a cast to same type 
// IGNORE_CLASS   - ignored dataset class used in line
// GEN_COMMENT    - replace this with a message about generated class
@SuppressWarnings("cast")
'''

class transmutate(object):
    def __init__(self, scriptfile, srcclass, source, dstclass, destination, disreal=True,
    disbool=False):
        '''
        scriptfile
        srcclass
        source
        dstclass
        destination
        disreal indicates whether destination is a real dataset
        disbool indicates whether destination is a boolean dataset

        source and destination are lists of strings which describe dtype,
        Java boxed primitive class, Java primitive type, getElement abstract method,
        Object converter toReal, string format, default expansion value
        (from class constant)
        '''
        self.sdsclass = srcclass
        self.ddsclass = dstclass
        self.commentline = "// Produced from %s.java by %s" % (srcclass, scriptfile)

        if len(source) != len(destination):
            raise ValueError, "length of lists should be the same"

        (self.sdtype, self.spclass, self.sprim, self.sgetel,
        self.sconv, self.sform, self.sdef) = source
        (self.ddtype, self.dpclass, self.dprim, self.dgetel,
        self.dconv, self.dform, self.ddef) = destination

        self.dcast = "(" + self.dprim + ") "

        if self.ddtype.startswith("INT"):
            self.dprimlong = self.dprim + ") (long"
        else:
            self.dprimlong = self.dprim

        self.isreal = disreal
        self.isbool = disbool
        if self.isbool:
            self.isreal = False

        self.processors = { "// DATA_TYPE": self.data,
            "// CLASS_TYPE": self.jpclass,
            "// PRIM_TYPE": self.primitive,
            "// PRIM_TYPE_LONG": self.primitivelong,
            "// GET_ELEMENT": self.getelement,
            "// FROM_OBJECT": self.fromobj,
            "// REAL_ONLY": self.unrealomit,
            "// BOOLEAN_OMIT": self.boolomit,
            "// BOOLEAN_USE": self.booluse,
            "// BOOLEAN_FALSE": self.boolfalse,
            "// BOOLEAN_ZERO": self.boolzero,
            "// FORMAT_STRING": self.string,
            "// INT_EXCEPTION": self.intexception,
            "// INT_ZEROTEST": self.intzerotest,
            "// OMIT_SAME_CAST": self.omitcast,
            "// DEFAULT_VAL": self.defval,
            "@SuppressWarnings(\"cast\")": self.omit }
        # also // IGNORE_CLASS
        self.processors[srcclass] = self.jclass

    def data(self, line):
        '''
        dataset type
        '''
        return line.replace(self.sdtype, self.ddtype)

    def jclass(self, line):
        '''
        dataset name is also used as Java class name
        '''
        return line.replace(self.sdsclass, self.ddsclass)

    def jpclass(self, line):
        '''
        Java class name for boxed primitive
        '''
        return line.replace(self.spclass, self.dpclass)

    def primitive(self, line):
        '''
        java primitive type is an element type
        '''
        return line.replace(self.sprim, self.dprim)

    def primitivelong(self, line):
        return line.replace(self.sprim, self.dprimlong)

    def getelement(self, line):
        return line.replace(self.sgetel, self.dgetel)

    def fromobj(self, line):
        return line.replace(self.sconv, self.dconv)

    def omitcast(self, line):
        return line.replace(self.dcast, "")

    def omit(self, line):
        return None

    def unrealomit(self, line):
        if not self.isreal:
            return None
        return line

    def boolomit(self, line):
        if self.isbool:
            return None
        return line.replace(" // BOOLEAN_OMIT", "")

    def booluse(self, line):
        if self.ddtype.startswith("BOOL"): # uncomment line
            s = line.find("// ")
            return line[:s] + line[s+3:]
        else:
            return line

    def boolfalse(self, line):
        if self.isbool:
            return "\t\treturn false;"
        return line

    def boolzero(self, line):
        if self.isbool:
            return "\t\treturn 0;"
        return line

    def string(self, line):
        return line.replace(self.sform, self.dform)

    def intexception(self, line):
        if self.ddtype.startswith("INT"):
            nformat = "\t\t\t\ttry {\n\t%s\n\t\t\t\t} catch (ArithmeticException e) {\n\t\t\t\t\t%s] = 0;\n\t\t\t\t}"
            lhs = line.split('] ')[0].lstrip()
            return nformat % (line, lhs)
        elif self.ddtype.startswith("ARRAYINT"):
            nformat = "\t\t\t\t\t\ttry {\n\t%s\n\t\t\t\t\t\t} catch (ArithmeticException e) {\n\t\t\t\t\t\t\t%s] = 0;\n\t\t\t\t\t\t}"
            lhs = line.split('] ')[0].lstrip()
            return nformat % (line, lhs)
        else:
            return line

    def intzerotest(self, line):
        if self.ddtype.startswith("INT") or self.ddtype.startswith("ARRAYINT"):
            s = line.find("// ")
            return line[:s] + line[s+3:]
        else:
            return line

    def defval(self, line):
        return line.replace(self.sdef, self.ddef)

    def processline(self, line):
        '''
        return processed line
        '''
        l = line.rstrip()
        if l.find("// IGNORE_CLASS") >= 0:
            return l
        for m in self.processors:
            if l == None or len(l) == 0:
                break
            if l.find(m) >= 0:
                p = self.processors[m]
                l = p(l)
        if l != None:
            if l.find("// GEN_COMMENT") >= 0:
                return self.commentline
        return l
