'''
Core package contains wrappers for Java dataset classes
'''

import uk.ac.diamond.scisoft.analysis.dataset.AbstractDataset as _abstractds
import uk.ac.diamond.scisoft.analysis.dataset.AbstractCompoundDataset as _abscompoundds

import uk.ac.diamond.scisoft.analysis.dataset.BooleanDataset as _booleands
import uk.ac.diamond.scisoft.analysis.dataset.ByteDataset as _byteds
import uk.ac.diamond.scisoft.analysis.dataset.ShortDataset as _shortds
import uk.ac.diamond.scisoft.analysis.dataset.IntegerDataset as _integerds
import uk.ac.diamond.scisoft.analysis.dataset.LongDataset as _longds
import uk.ac.diamond.scisoft.analysis.dataset.CompoundByteDataset as _compoundbyteds
import uk.ac.diamond.scisoft.analysis.dataset.CompoundShortDataset as _compoundshortds
import uk.ac.diamond.scisoft.analysis.dataset.CompoundIntegerDataset as _compoundintegerds
import uk.ac.diamond.scisoft.analysis.dataset.CompoundLongDataset as _compoundlongds
import uk.ac.diamond.scisoft.analysis.dataset.FloatDataset as _floatds
import uk.ac.diamond.scisoft.analysis.dataset.DoubleDataset as _doubleds
import uk.ac.diamond.scisoft.analysis.dataset.CompoundFloatDataset as _compoundfloatds
import uk.ac.diamond.scisoft.analysis.dataset.CompoundDoubleDataset as _compounddoubleds
import uk.ac.diamond.scisoft.analysis.dataset.RGBDataset as _rgbds
import uk.ac.diamond.scisoft.analysis.dataset.ComplexFloatDataset as _complexfloatds
import uk.ac.diamond.scisoft.analysis.dataset.ComplexDoubleDataset as _complexdoubleds

import uk.ac.diamond.scisoft.analysis.dataset.DatasetUtils as _dsutils

from gda.analysis import DataSet as _dataset

import Jama.Matrix as _matrix

import scisoftpy.maths as _maths

import types as _types

import java.lang.ArrayIndexOutOfBoundsException as _jarrayindex_exception

bool = _abstractds.BOOL
int8 = _abstractds.INT8
int16 = _abstractds.INT16
int32 = _abstractds.INT32
int64 = _abstractds.INT64
cint8 = _abstractds.ARRAYINT8
cint16 = _abstractds.ARRAYINT16
cint32 = _abstractds.ARRAYINT32
cint64 = _abstractds.ARRAYINT64
float32 = _abstractds.FLOAT32
float64 = _abstractds.FLOAT64
cfloat32 = _abstractds.ARRAYFLOAT32
cfloat64 = _abstractds.ARRAYFLOAT64
complex64 = _abstractds.COMPLEX64
complex128 = _abstractds.COMPLEX128

_pyint = int
_pyfloat = float
_pycomplex = complex
int_ = int64
int = int_
float_ = float64
float = float_
complex_ = complex128
complex = complex_

from jarray import array as _array
_arraytype = type(_array([0], 'f')) # this is used for testing if returned object is a Java array

import java.util.List as _jlist
import java.util.Map as _jmap

def Sciwrap(a):
    """
    This wrapper function is required for any Java method that returns a dataset
    """
    if a is None:
        raise ValueError, "No value given"
    if isinstance(a, _rgbds):
        return ndarrayRGB(a, True)
    if isinstance(a, _abscompoundds):
        return __cdtype2jythoncls[a.dtype](a, True)
    if isinstance(a, _abstractds):
        return __dtype2jythoncls[a.dtype](a, True)
    return a

def Sciunwrap(a):
    """
    This unwrapping function is required to pass back a dataset across CORBA
    """
    if a is None:
        raise ValueError, "No value given"
    if isinstance(a, _rgbds):
        return _rgbds(a, True)
    if isinstance(a, _abscompoundds):
        return __cdtype2javacls[a.dtype](a, True)
    if isinstance(a, _abstractds):
        return __dtype2javacls[a.dtype](a, True)
    return a

ndarraywrapped = _maths.ndarraywrapped

def asIterable(items):
    """
    Ensure entity is an iterable by making it a tuple if not
    """
    t = type(items)
    if t is _types.ListType or t is _types.TupleType or t is _arraytype:
        pass
    elif isinstance(items, _jlist) or isinstance(items, _jmap):
        pass
    else: # isinstance(items, _abstractds) or isinstance(items, _dataset):
        items = (items,)
    return items

def toList(listdata):
    '''Convert a list or tuple to list of datasets'''
    return [ d for d in asIterable(listdata) ]

def toUnwrappedList(listdata):
    '''Convert a list or tuple to list of unwrapped datasets (for CORBA transport)'''
    return [ Sciunwrap(d) for d in asIterable(listdata) ]

def toDS(data):
    '''Convert to DataSet, if necessary'''
    if isinstance(data, _dataset):
        return data
    else:
        if isinstance(data, _abstractds):
            cdata = data.cast(float64)
            return _abstractds.toDataSet(cdata)
        else:
            raise ValueError, "Not a dataset"

def fromDS(data):
    '''Convert from a DataSet'''
    if isinstance(data, _dataset):
        return Sciwrap(_abstractds.toDoubleDataset(data))
    elif isinstance(data, _abstractds):
        return Sciwrap(data)
    return data

def asDataset(data, dtype=None):
    """
    Used for arithmetic ops to coerce a sequence to a dataset otherwise leave as single item
    """
    if isinstance(data, _abstractds):
        return data
    if isinstance(data, _dataset):
        return Sciwrap(_abstractds.toDoubleDataset(data))

    try:
        iter(data)
    except:
        return data

    return array(data, dtype)

asarray = asDataset
asanyarray = asDataset

def asDatasetList(dslist):
    """
    Used to coerce a list of DataSets to a list of datasets
    """
    return [ fromDS(d) for d in asIterable(dslist) ]

def asDatasetDict(dsdict):
    """
    Used to coerce a dictionary of DataSets to a dictionary of datasets
    """
    rdict = {}
    for k in dsdict:
        rdict[k] = fromDS(dsdict[k])
    return rdict

def _isslice(rank, shape, key):
#    print rank, shape, key
    key = asIterable(key)
    if rank < len(shape):
        raise IndexError, "Rank and shape do not correspond"

    if len(key) < rank:
        return True
    elif len(key) > rank:
        raise IndexError, "Too many indices"

    for k in key:
        if isinstance(k, slice):
            return True
    return False

isslice = _isslice

import scisoftpy.comparisons as _cmps

class ndarray:
    """
    Class to hold special methods and non-overloading names
    """
    def __add__(self, o):
        return _maths.add(self, asDataset(o))
    def __radd__(self, o):
        return _maths.add(self, asDataset(o))
    def __iadd__(self, o):
        return self.iadd(asDataset(o, self.dtype))

    def __sub__(self, o):
        return _maths.subtract(self, asDataset(o))
    def __rsub__(self, o):
        return _maths.subtract(asDataset(o), self)
    def __isub__(self, o):
        return self.isubtract(asDataset(o, self.dtype))

    def __mul__(self, o):
        return _maths.multiply(self, asDataset(o))
    def __rmul__(self, o):
        return _maths.multiply(self, asDataset(o))
    def __imul__(self, o):
        return self.imultiply(asDataset(o, self.dtype))

    def __div__(self, o):
        return _maths.divide(self, asDataset(o))
    def __rdiv__(self, o):
        return _maths.divide(asDataset(o), self)
    def __idiv__(self, o):
        return self.idivide(asDataset(o, self.dtype))

    def __truediv__(self, o):
        return _maths.divide(self, asDataset(o))
    def __itruediv__(self, o):
        return self.idivide(asDataset(o, self.dtype))

    def __floordiv__(self, o):
        return _maths.floor_divide(self, asDataset(o))
    def __ifloordiv__(self, o):
        return self.ifloordivide(asDataset(o, self.dtype))

    def __mod__(self, o):
        return _maths.remainder(self, asDataset(o))
    def __imod__(self, o):
        return self.iremainder(asDataset(o, self.dtype))

    def __neg__(self):
        return _maths.negative(self)
    def __pos__(self):
        return self
    def __pow__(self, o):
        return _maths.power(self, asDataset(o))
    def __ipow__(self, o):
        return self.ipower(asDataset(o, self.dtype))

    def __eq__(self, o):
        return _cmps.equal(self, o)

    def __ne__(self, o):
        return _cmps.not_equal(self, o)

    def __lt__(self, o):
        return _cmps.less(self, o)

    def __le__(self, o):
        return _cmps.less_equal(self, o)

    def __gt__(self, o):
        return _cmps.greater(self, o)

    def __ge__(self, o):
        return _cmps.greater_equal(self, o)


    def __len__(self):
        if len(self.shape) > 0:
            return self.shape[0]
        return 0

    @ndarraywrapped
    def __copy__(self):
        return self.clone()

    def _toslice(self, key):
        '''Transform key to proper slice if necessary
        '''
        if self.rank == 1:
            if isinstance(key, slice):
                return True, key
            if isinstance(key, tuple):
                if len(key) > 1:
                    raise IndexError, "too many indices"
                return False, key
            return False, (key,)

        if _isslice(self.rank, self.shape, key):
            return True, key
        return False, key

    @ndarraywrapped
    def __getitem__(self, key):
        if isinstance(key, _booleands):
            return self.getByBoolean(key)
        if isinstance(key, _integerds):
            return self.getByIndex(key)

        isslice, key = self._toslice(key)
        try:
            if isslice:
                return self.getSlice(key)
            return self.getObject(key)
        except _jarrayindex_exception:
            raise IndexError

    def __setitem__(self, key, value):
        value = fromDS(value)
        if isinstance(key, _booleands):
            return self.setByBoolean(key, value)
        if isinstance(key, _integerds):
            return self.setByIndex(key, value)

        isslice, key = self._toslice(key)
        try:
            if isslice:
                return self.setSlice(key, value)
            return self.set(value, key)
        except _jarrayindex_exception:
            raise IndexError

    def __iter__(self):
        def ndgen(d):
            iterator = d.getIterator()
            while iterator.hasNext():
                yield d.getObjectAbs(iterator.index)
        return ndgen(self)

    @ndarraywrapped
    def index(self):
        return self.getIndexDataset()
#
    def conj(self):
        return _maths.conj(self)

    @ndarraywrapped
    def ravel(self):
        return self.flatten()

    def isnan(self):
        return self.containsNans()

    def isinf(self):
        return self.containsInfs()

    def amax(self, axis=None):
        return self.max(axis)

    def amin(self, axis=None):
        return self.min(axis)

    def prod(self, axis=None):
        return _maths.prod(self, axis)

    @ndarraywrapped
    def var(self, axis=None, ddof=0):
        if ddof == 1:
            if axis == None:
                return self.variance()
            else:
                return self.variance(axis)
        else:
            if axis == None:
                v = self.variance()
                n = self.getStoredValue("stats").getN()
            else:
                v = Sciwrap(self.variance(axis))
                n = Sciwrap(self.getStoredValue("count-%d" % axis))
            f = (n - 1.)/(n - ddof)
            return v * f

    @ndarraywrapped
    def std(self, axis=None, ddof=0):
        if ddof == 1:
            if axis == None:
                return self.stdDeviation()
            else:
                return self.stdDeviation(axis)
        else:
            if axis == None:
                s = self.stdDeviation()
                n = self.getStoredValue("stats").getN()
            else:
                s = Sciwrap(self.stdDeviation(axis))
                n = Sciwrap(self.getStoredValue("count-%d" % axis))
            f = _maths.sqrt((n - 1.)/(n - ddof))
            return s * f

    @ndarraywrapped
    def rms(self, axis=None):
        if axis == None:
            return self.rootMeanSquare()
        else:
            return self.rootMeanSquare(axis)

    @ndarraywrapped
    def ptp(self, axis=None):
        if axis == None:
            return self.peakToPeak()
        else:
            return self.peakToPeak(axis)

    def clip(self, a_min, a_max):
        return _maths.clip(self, a_min, a_max)

    @ndarraywrapped
    def argmax(self, axis=None):
        if axis == None:
            return self.argMax()
        else:
            return self.argMax(axis)

    @ndarraywrapped
    def argmin(self, axis=None):
        if axis == None:
            return self.argMin()
        else:
            return self.argMin(axis)

    # properties
    @ndarraywrapped
    def _get_transpose(self):
        return self.transpose()

    T = property(_get_transpose)

    @ndarraywrapped
    def _get_view(self):
        '''Return a view of dataset'''
        return self.getView()

    view = property(_get_view)

    @ndarraywrapped
    def _get_indices(self):
        '''Return an index dataset'''
        return self.getIndices()

    indices = property(_get_indices)

    def _set_shape(self, *shape):
        if len(shape) == 1:
            shape = asIterable(shape[0])
        self.setShape(shape)

    def _get_shape(self):
        return self.getShape()

    shape = property(_get_shape, _set_shape)

class ndarrayA(ndarray, _booleands):
    """
    Wrap boolean dataset
    """
    def __init__(self, *arg):
        _booleands.__init__(self, *arg) #@UndefinedVariable
        self.ndcls = _booleands

    def copy(self): # override to keep superclass's methods
        return ndarrayA(self)

    # non-specific code that needs ndcls
    # this code cannot put in ndarray superclass as there is a problem when
    # wrapping methods with same name in superclass
    @ndarraywrapped
    def max(self, axis=None):
        if axis == None:
            return self.ndcls.max(self)
        else:
            return self.ndcls.max(self, axis)

    @ndarraywrapped
    def min(self, axis=None):
        if axis == None:
            return self.ndcls.min(self)
        else:
            return self.ndcls.min(self, axis)

    @ndarraywrapped
    def sum(self, axis=None):
        if axis == None:
            return self.ndcls.sum(self)
        else:
            return self.ndcls.sum(self, axis)

    @ndarraywrapped
    def mean(self, axis=None):
        if axis == None:
            return self.ndcls.mean(self)
        else:
            return self.ndcls.mean(self, axis)

    def sort(self, axis=-1):
        return self.ndcls.sort(self, axis)

    @ndarraywrapped
    def transpose(self, axes=None):
        if axes == None:
            axes = ()
        return self.ndcls.transpose(self, asIterable(axes))

    def put(self, indices, values):
        return self.ndcls.put(self, asIterable(indices), asIterable(values))

    @ndarraywrapped
    def take(self, indices, axis=None):
        return self.ndcls.take(self, asIterable(indices), axis)

    @ndarraywrapped
    def all(self, axis=None):
        if axis == None:
            return self.ndcls.all(self)
        else:
            return self.ndcls.all(self, axis)

    @ndarraywrapped
    def any(self, axis=None):
        if axis == None:
            return self.ndcls.any(self)
        else:
            return self.ndcls.any(self, axis)

    def append(self, other, axis=-1):
        return self.ndcls.append(self, other, axis)

    @ndarraywrapped
    def reshape(self, *shape):
        '''Return a dataset with same data but new shape'''
        if len(shape) == 1:
            shape = asIterable(shape[0])
        return self.ndcls.reshape(self, shape)


class ndarrayB(ndarray, _byteds):
    """
    Wrap byte dataset
    """
    def __init__(self, *arg):
        _byteds.__init__(self, *arg) #@UndefinedVariable
        self.ndcls = _byteds

    def copy(self): # override to keep superclass's methods
        return ndarrayB(self)

    # non-specific code that needs ndcls
    # this code cannot put in ndarray superclass as there is a problem when
    # wrapping methods with same name in superclass
    @ndarraywrapped
    def max(self, axis=None):
        if axis == None:
            return self.ndcls.max(self)
        else:
            return self.ndcls.max(self, axis)

    @ndarraywrapped
    def min(self, axis=None):
        if axis == None:
            return self.ndcls.min(self)
        else:
            return self.ndcls.min(self, axis)

    @ndarraywrapped
    def sum(self, axis=None):
        if axis == None:
            return self.ndcls.sum(self)
        else:
            return self.ndcls.sum(self, axis)

    @ndarraywrapped
    def mean(self, axis=None):
        if axis == None:
            return self.ndcls.mean(self)
        else:
            return self.ndcls.mean(self, axis)

    def sort(self, axis=-1):
        return self.ndcls.sort(self, axis)

    @ndarraywrapped
    def transpose(self, axes=None):
        if axes == None:
            axes = ()
        return self.ndcls.transpose(self, asIterable(axes))

    def put(self, indices, values):
        return self.ndcls.put(self, asIterable(indices), asIterable(values))

    @ndarraywrapped
    def take(self, indices, axis=None):
        return self.ndcls.take(self, asIterable(indices), axis)

    @ndarraywrapped
    def all(self, axis=None):
        if axis == None:
            return self.ndcls.all(self)
        else:
            return self.ndcls.all(self, axis)

    @ndarraywrapped
    def any(self, axis=None):
        if axis == None:
            return self.ndcls.any(self)
        else:
            return self.ndcls.any(self, axis)

    def append(self, other, axis=-1):
        return self.ndcls.append(self, other, axis)

    @ndarraywrapped
    def reshape(self, *shape):
        '''Return a dataset with same data but new shape'''
        if len(shape) == 1:
            shape = asIterable(shape[0])
        return self.ndcls.reshape(self, shape)


class ndarrayS(ndarray, _shortds):
    """
    Wrap short dataset
    """
    def __init__(self, *arg):
        _shortds.__init__(self, *arg) #@UndefinedVariable
        self.ndcls = _shortds

    def copy(self): # override to keep superclass's methods
        return ndarrayS(self)

    # non-specific code that needs ndcls
    # this code cannot put in ndarray superclass as there is a problem when
    # wrapping methods with same name in superclass
    @ndarraywrapped
    def max(self, axis=None):
        if axis == None:
            return self.ndcls.max(self)
        else:
            return self.ndcls.max(self, axis)

    @ndarraywrapped
    def min(self, axis=None):
        if axis == None:
            return self.ndcls.min(self)
        else:
            return self.ndcls.min(self, axis)

    @ndarraywrapped
    def sum(self, axis=None):
        if axis == None:
            return self.ndcls.sum(self)
        else:
            return self.ndcls.sum(self, axis)

    @ndarraywrapped
    def mean(self, axis=None):
        if axis == None:
            return self.ndcls.mean(self)
        else:
            return self.ndcls.mean(self, axis)

    def sort(self, axis=-1):
        return self.ndcls.sort(self, axis)

    @ndarraywrapped
    def transpose(self, axes=None):
        if axes == None:
            axes = ()
        return self.ndcls.transpose(self, asIterable(axes))

    def put(self, indices, values):
        return self.ndcls.put(self, asIterable(indices), asIterable(values))

    @ndarraywrapped
    def take(self, indices, axis=None):
        return self.ndcls.take(self, asIterable(indices), axis)

    @ndarraywrapped
    def all(self, axis=None):
        if axis == None:
            return self.ndcls.all(self)
        else:
            return self.ndcls.all(self, axis)

    @ndarraywrapped
    def any(self, axis=None):
        if axis == None:
            return self.ndcls.any(self)
        else:
            return self.ndcls.any(self, axis)

    def append(self, other, axis=-1):
        return self.ndcls.append(self, other, axis)

    @ndarraywrapped
    def reshape(self, *shape):
        '''Return a dataset with same data but new shape'''
        if len(shape) == 1:
            shape = asIterable(shape[0])
        return self.ndcls.reshape(self, shape)


class ndarrayI(ndarray, _integerds):
    """
    Wrap integer dataset
    """
    def __init__(self, *arg):
        _integerds.__init__(self, *arg) #@UndefinedVariable
        self.ndcls = _integerds

    def copy(self): # override to keep superclass's methods
        return ndarrayI(self)

    # non-specific code that needs ndcls
    # this code cannot put in ndarray superclass as there is a problem when
    # wrapping methods with same name in superclass
    @ndarraywrapped
    def max(self, axis=None):
        if axis == None:
            return self.ndcls.max(self)
        else:
            return self.ndcls.max(self, axis)

    @ndarraywrapped
    def min(self, axis=None):
        if axis == None:
            return self.ndcls.min(self)
        else:
            return self.ndcls.min(self, axis)

    @ndarraywrapped
    def sum(self, axis=None):
        if axis == None:
            return self.ndcls.sum(self)
        else:
            return self.ndcls.sum(self, axis)

    @ndarraywrapped
    def mean(self, axis=None):
        if axis == None:
            return self.ndcls.mean(self)
        else:
            return self.ndcls.mean(self, axis)

    def sort(self, axis=-1):
        return self.ndcls.sort(self, axis)

    @ndarraywrapped
    def transpose(self, axes=None):
        if axes == None:
            axes = ()
        return self.ndcls.transpose(self, asIterable(axes))

    def put(self, indices, values):
        return self.ndcls.put(self, asIterable(indices), asIterable(values))

    @ndarraywrapped
    def take(self, indices, axis=None):
        return self.ndcls.take(self, asIterable(indices), axis)

    @ndarraywrapped
    def all(self, axis=None):
        if axis == None:
            return self.ndcls.all(self)
        else:
            return self.ndcls.all(self, axis)

    @ndarraywrapped
    def any(self, axis=None):
        if axis == None:
            return self.ndcls.any(self)
        else:
            return self.ndcls.any(self, axis)

    def append(self, other, axis=-1):
        return self.ndcls.append(self, other, axis)

    @ndarraywrapped
    def reshape(self, *shape):
        '''Return a dataset with same data but new shape'''
        if len(shape) == 1:
            shape = asIterable(shape[0])
        return self.ndcls.reshape(self, shape)


class ndarrayL(ndarray, _longds):
    """
    Wrap long dataset
    """
    def __init__(self, *arg):
        _longds.__init__(self, *arg) #@UndefinedVariable
        self.ndcls = _longds

    def copy(self): # override to keep superclass's methods
        return ndarrayL(self)

    # non-specific code that needs ndcls
    # this code cannot put in ndarray superclass as there is a problem when
    # wrapping methods with same name in superclass
    @ndarraywrapped
    def max(self, axis=None):
        if axis == None:
            return self.ndcls.max(self)
        else:
            return self.ndcls.max(self, axis)

    @ndarraywrapped
    def min(self, axis=None):
        if axis == None:
            return self.ndcls.min(self)
        else:
            return self.ndcls.min(self, axis)

    @ndarraywrapped
    def sum(self, axis=None):
        if axis == None:
            return self.ndcls.sum(self)
        else:
            return self.ndcls.sum(self, axis)

    @ndarraywrapped
    def mean(self, axis=None):
        if axis == None:
            return self.ndcls.mean(self)
        else:
            return self.ndcls.mean(self, axis)

    def sort(self, axis=-1):
        return self.ndcls.sort(self, axis)

    @ndarraywrapped
    def transpose(self, axes=None):
        if axes == None:
            axes = ()
        return self.ndcls.transpose(self, asIterable(axes))

    def put(self, indices, values):
        return self.ndcls.put(self, asIterable(indices), asIterable(values))

    @ndarraywrapped
    def take(self, indices, axis=None):
        return self.ndcls.take(self, asIterable(indices), axis)

    @ndarraywrapped
    def all(self, axis=None):
        if axis == None:
            return self.ndcls.all(self)
        else:
            return self.ndcls.all(self, axis)

    @ndarraywrapped
    def any(self, axis=None):
        if axis == None:
            return self.ndcls.any(self)
        else:
            return self.ndcls.any(self, axis)

    def append(self, other, axis=-1):
        return self.ndcls.append(self, other, axis)

    @ndarraywrapped
    def reshape(self, *shape):
        '''Return a dataset with same data but new shape'''
        if len(shape) == 1:
            shape = asIterable(shape[0])
        return self.ndcls.reshape(self, shape)


class ndarrayF(ndarray, _floatds):
    """
    Wrap float dataset
    """
    def __init__(self, *arg):
        _floatds.__init__(self, *arg) #@UndefinedVariable
        self.ndcls = _floatds

    def copy(self): # override to keep superclass's methods
        return ndarrayF(self)

    # non-specific code that needs ndcls
    # this code cannot put in ndarray superclass as there is a problem when
    # wrapping methods with same name in superclass
    @ndarraywrapped
    def max(self, axis=None):
        if axis == None:
            return self.ndcls.max(self)
        else:
            return self.ndcls.max(self, axis)

    @ndarraywrapped
    def min(self, axis=None):
        if axis == None:
            return self.ndcls.min(self)
        else:
            return self.ndcls.min(self, axis)

    @ndarraywrapped
    def sum(self, axis=None):
        if axis == None:
            return self.ndcls.sum(self)
        else:
            return self.ndcls.sum(self, axis)

    @ndarraywrapped
    def mean(self, axis=None):
        if axis == None:
            return self.ndcls.mean(self)
        else:
            return self.ndcls.mean(self, axis)

    def sort(self, axis=-1):
        return self.ndcls.sort(self, axis)

    @ndarraywrapped
    def transpose(self, axes=None):
        if axes == None:
            axes = ()
        return self.ndcls.transpose(self, asIterable(axes))

    def put(self, indices, values):
        return self.ndcls.put(self, asIterable(indices), asIterable(values))

    @ndarraywrapped
    def take(self, indices, axis=None):
        return self.ndcls.take(self, asIterable(indices), axis)

    @ndarraywrapped
    def all(self, axis=None):
        if axis == None:
            return self.ndcls.all(self)
        else:
            return self.ndcls.all(self, axis)

    @ndarraywrapped
    def any(self, axis=None):
        if axis == None:
            return self.ndcls.any(self)
        else:
            return self.ndcls.any(self, axis)

    def append(self, other, axis=-1):
        return self.ndcls.append(self, other, axis)

    @ndarraywrapped
    def reshape(self, *shape):
        '''Return a dataset with same data but new shape'''
        if len(shape) == 1:
            shape = asIterable(shape[0])
        return self.ndcls.reshape(self, shape)


class ndarrayD(ndarray, _doubleds):
    """
    Wrap double dataset
    """
    def __init__(self, *arg):
        _doubleds.__init__(self, *arg) #@UndefinedVariable
        self.ndcls = _doubleds

    def copy(self): # override to keep superclass's methods
        return ndarrayD(self)

    # non-specific code that needs ndcls
    # this code cannot put in ndarray superclass as there is a problem when
    # wrapping methods with same name in superclass
    @ndarraywrapped
    def max(self, axis=None):
        if axis == None:
            return self.ndcls.max(self)
        else:
            return self.ndcls.max(self, axis)

    @ndarraywrapped
    def min(self, axis=None):
        if axis == None:
            return self.ndcls.min(self)
        else:
            return self.ndcls.min(self, axis)

    @ndarraywrapped
    def sum(self, axis=None):
        if axis == None:
            return self.ndcls.sum(self)
        else:
            return self.ndcls.sum(self, axis)

    @ndarraywrapped
    def mean(self, axis=None):
        if axis == None:
            return self.ndcls.mean(self)
        else:
            return self.ndcls.mean(self, axis)

    def sort(self, axis=-1):
        return self.ndcls.sort(self, axis)

    @ndarraywrapped
    def transpose(self, axes=None):
        if axes == None:
            axes = ()
        return self.ndcls.transpose(self, asIterable(axes))

    def put(self, indices, values):
        return self.ndcls.put(self, asIterable(indices), asIterable(values))

    @ndarraywrapped
    def take(self, indices, axis=None):
        return self.ndcls.take(self, asIterable(indices), axis)

    @ndarraywrapped
    def all(self, axis=None):
        if axis == None:
            return self.ndcls.all(self)
        else:
            return self.ndcls.all(self, axis)

    @ndarraywrapped
    def any(self, axis=None):
        if axis == None:
            return self.ndcls.any(self)
        else:
            return self.ndcls.any(self, axis)

    def append(self, other, axis=-1):
        return self.ndcls.append(self, other, axis)

    @ndarraywrapped
    def reshape(self, *shape):
        '''Return a dataset with same data but new shape'''
        if len(shape) == 1:
            shape = asIterable(shape[0])
        return self.ndcls.reshape(self, shape)


class ndarrayC(ndarray, _complexfloatds):
    """
    Wrap complex float dataset
    """
    def __init__(self, *arg):
        _complexfloatds.__init__(self, *arg) #@UndefinedVariable
        self.ndcls = _complexfloatds

    def copy(self): # override to keep superclass's methods
        return ndarrayC(self)

    # non-specific code that needs ndcls
    # this code cannot put in ndarray superclass as there is a problem when
    # wrapping methods with same name in superclass
    @ndarraywrapped
    def max(self, axis=None):
        if axis == None:
            return self.ndcls.max(self)
        else:
            return self.ndcls.max(self, axis)

    @ndarraywrapped
    def min(self, axis=None):
        if axis == None:
            return self.ndcls.min(self)
        else:
            return self.ndcls.min(self, axis)

    @ndarraywrapped
    def sum(self, axis=None):
        if axis == None:
            return self.ndcls.sum(self)
        else:
            return self.ndcls.sum(self, axis)

    @ndarraywrapped
    def mean(self, axis=None):
        if axis == None:
            return self.ndcls.mean(self)
        else:
            return self.ndcls.mean(self, axis)

    def sort(self, axis=-1):
        return self.ndcls.sort(self, axis)

    @ndarraywrapped
    def transpose(self, axes=None):
        if axes == None:
            axes = ()
        return self.ndcls.transpose(self, asIterable(axes))

    def put(self, indices, values):
        return self.ndcls.put(self, asIterable(indices), asIterable(values))

    @ndarraywrapped
    def take(self, indices, axis=None):
        return self.ndcls.take(self, asIterable(indices), axis)

    @ndarraywrapped
    def all(self, axis=None):
        if axis == None:
            return self.ndcls.all(self)
        else:
            return self.ndcls.all(self, axis)

    @ndarraywrapped
    def any(self, axis=None):
        if axis == None:
            return self.ndcls.any(self)
        else:
            return self.ndcls.any(self, axis)

    def append(self, other, axis=-1):
        return self.ndcls.append(self, other, axis)

    @ndarraywrapped
    def reshape(self, *shape):
        '''Return a dataset with same data but new shape'''
        if len(shape) == 1:
            shape = asIterable(shape[0])
        return self.ndcls.reshape(self, shape)


class ndarrayZ(ndarray, _complexdoubleds):
    """
    Wrap complex double dataset
    """
    def __init__(self, *arg):
        _complexdoubleds.__init__(self, *arg) #@UndefinedVariable
        self.ndcls = _complexdoubleds

    def copy(self): # override to keep superclass's methods
        return ndarrayZ(self)

    # non-specific code that needs ndcls
    # this code cannot put in ndarray superclass as there is a problem when
    # wrapping methods with same name in superclass
    @ndarraywrapped
    def max(self, axis=None):
        if axis == None:
            return self.ndcls.max(self)
        else:
            return self.ndcls.max(self, axis)

    @ndarraywrapped
    def min(self, axis=None):
        if axis == None:
            return self.ndcls.min(self)
        else:
            return self.ndcls.min(self, axis)

    @ndarraywrapped
    def sum(self, axis=None):
        if axis == None:
            return self.ndcls.sum(self)
        else:
            return self.ndcls.sum(self, axis)

    @ndarraywrapped
    def mean(self, axis=None):
        if axis == None:
            return self.ndcls.mean(self)
        else:
            return self.ndcls.mean(self, axis)

    def sort(self, axis=-1):
        return self.ndcls.sort(self, axis)

    @ndarraywrapped
    def transpose(self, axes=None):
        if axes == None:
            axes = ()
        return self.ndcls.transpose(self, asIterable(axes))

    def put(self, indices, values):
        return self.ndcls.put(self, asIterable(indices), asIterable(values))

    @ndarraywrapped
    def take(self, indices, axis=None):
        return self.ndcls.take(self, asIterable(indices), axis)

    @ndarraywrapped
    def all(self, axis=None):
        if axis == None:
            return self.ndcls.all(self)
        else:
            return self.ndcls.all(self, axis)

    @ndarraywrapped
    def any(self, axis=None):
        if axis == None:
            return self.ndcls.any(self)
        else:
            return self.ndcls.any(self, axis)

    def append(self, other, axis=-1):
        return self.ndcls.append(self, other, axis)

    @ndarraywrapped
    def reshape(self, *shape):
        '''Return a dataset with same data but new shape'''
        if len(shape) == 1:
            shape = asIterable(shape[0])
        return self.ndcls.reshape(self, shape)

class ndarrayCB(ndarray, _compoundbyteds):
    """
    Wrap compound byte dataset
    """
    def __init__(self, *arg):
        _compoundbyteds.__init__(self, *arg) #@UndefinedVariable
        self.ndcls = _compoundbyteds

    def copy(self): # override to keep superclass's methods
        return ndarrayCB(self)

    # non-specific code that needs ndcls
    # this code cannot put in ndarray superclass as there is a problem when
    # wrapping methods with same name in superclass
    @ndarraywrapped
    def max(self, axis=None):
        if axis == None:
            return self.ndcls.max(self)
        else:
            return self.ndcls.max(self, axis)

    @ndarraywrapped
    def min(self, axis=None):
        if axis == None:
            return self.ndcls.min(self)
        else:
            return self.ndcls.min(self, axis)

    @ndarraywrapped
    def sum(self, axis=None):
        if axis == None:
            return self.ndcls.sum(self)
        else:
            return self.ndcls.sum(self, axis)

    @ndarraywrapped
    def mean(self, axis=None):
        if axis == None:
            return self.ndcls.mean(self)
        else:
            return self.ndcls.mean(self, axis)

    def sort(self, axis=-1):
        return self.ndcls.sort(self, axis)

    @ndarraywrapped
    def transpose(self, axes=None):
        if axes == None:
            axes = ()
        return self.ndcls.transpose(self, asIterable(axes))

    def put(self, indices, values):
        return self.ndcls.put(self, asIterable(indices), asIterable(values))

    @ndarraywrapped
    def take(self, indices, axis=None):
        return self.ndcls.take(self, asIterable(indices), axis)

    @ndarraywrapped
    def all(self, axis=None):
        if axis == None:
            return self.ndcls.all(self)
        else:
            return self.ndcls.all(self, axis)

    @ndarraywrapped
    def any(self, axis=None):
        if axis == None:
            return self.ndcls.any(self)
        else:
            return self.ndcls.any(self, axis)

    def append(self, other, axis=-1):
        return self.ndcls.append(self, other, axis)

    @ndarraywrapped
    def reshape(self, *shape):
        '''Return a dataset with same data but new shape'''
        if len(shape) == 1:
            shape = asIterable(shape[0])
        return self.ndcls.reshape(self, shape)

class ndarrayCS(ndarray, _compoundshortds):
    """
    Wrap compound short dataset
    """
    def __init__(self, *arg):
        _compoundshortds.__init__(self, *arg) #@UndefinedVariable
        self.ndcls = _compoundshortds

    def copy(self): # override to keep superclass's methods
        return ndarrayCS(self)

    # non-specific code that needs ndcls
    # this code cannot put in ndarray superclass as there is a problem when
    # wrapping methods with same name in superclass
    @ndarraywrapped
    def max(self, axis=None):
        if axis == None:
            return self.ndcls.max(self)
        else:
            return self.ndcls.max(self, axis)

    @ndarraywrapped
    def min(self, axis=None):
        if axis == None:
            return self.ndcls.min(self)
        else:
            return self.ndcls.min(self, axis)

    @ndarraywrapped
    def sum(self, axis=None):
        if axis == None:
            return self.ndcls.sum(self)
        else:
            return self.ndcls.sum(self, axis)

    @ndarraywrapped
    def mean(self, axis=None):
        if axis == None:
            return self.ndcls.mean(self)
        else:
            return self.ndcls.mean(self, axis)

    def sort(self, axis=-1):
        return self.ndcls.sort(self, axis)

    @ndarraywrapped
    def transpose(self, axes=None):
        if axes == None:
            axes = ()
        return self.ndcls.transpose(self, asIterable(axes))

    def put(self, indices, values):
        return self.ndcls.put(self, asIterable(indices), asIterable(values))

    @ndarraywrapped
    def take(self, indices, axis=None):
        return self.ndcls.take(self, asIterable(indices), axis)

    @ndarraywrapped
    def all(self, axis=None):
        if axis == None:
            return self.ndcls.all(self)
        else:
            return self.ndcls.all(self, axis)

    @ndarraywrapped
    def any(self, axis=None):
        if axis == None:
            return self.ndcls.any(self)
        else:
            return self.ndcls.any(self, axis)

    def append(self, other, axis=-1):
        return self.ndcls.append(self, other, axis)

    @ndarraywrapped
    def reshape(self, *shape):
        '''Return a dataset with same data but new shape'''
        if len(shape) == 1:
            shape = asIterable(shape[0])
        return self.ndcls.reshape(self, shape)

class ndarrayCI(ndarray, _compoundintegerds):
    """
    Wrap compound integer dataset
    """
    def __init__(self, *arg):
        _compoundintegerds.__init__(self, *arg) #@UndefinedVariable
        self.ndcls = _compoundintegerds

    def copy(self): # override to keep superclass's methods
        return ndarrayCI(self)

    # non-specific code that needs ndcls
    # this code cannot put in ndarray superclass as there is a problem when
    # wrapping methods with same name in superclass
    @ndarraywrapped
    def max(self, axis=None):
        if axis == None:
            return self.ndcls.max(self)
        else:
            return self.ndcls.max(self, axis)

    @ndarraywrapped
    def min(self, axis=None):
        if axis == None:
            return self.ndcls.min(self)
        else:
            return self.ndcls.min(self, axis)

    @ndarraywrapped
    def sum(self, axis=None):
        if axis == None:
            return self.ndcls.sum(self)
        else:
            return self.ndcls.sum(self, axis)

    @ndarraywrapped
    def mean(self, axis=None):
        if axis == None:
            return self.ndcls.mean(self)
        else:
            return self.ndcls.mean(self, axis)

    def sort(self, axis=-1):
        return self.ndcls.sort(self, axis)

    @ndarraywrapped
    def transpose(self, axes=None):
        if axes == None:
            axes = ()
        return self.ndcls.transpose(self, asIterable(axes))

    def put(self, indices, values):
        return self.ndcls.put(self, asIterable(indices), asIterable(values))

    @ndarraywrapped
    def take(self, indices, axis=None):
        return self.ndcls.take(self, asIterable(indices), axis)

    @ndarraywrapped
    def all(self, axis=None):
        if axis == None:
            return self.ndcls.all(self)
        else:
            return self.ndcls.all(self, axis)

    @ndarraywrapped
    def any(self, axis=None):
        if axis == None:
            return self.ndcls.any(self)
        else:
            return self.ndcls.any(self, axis)

    def append(self, other, axis=-1):
        return self.ndcls.append(self, other, axis)

    @ndarraywrapped
    def reshape(self, *shape):
        '''Return a dataset with same data but new shape'''
        if len(shape) == 1:
            shape = asIterable(shape[0])
        return self.ndcls.reshape(self, shape)

class ndarrayCL(ndarray, _compoundlongds):
    """
    Wrap compound long dataset
    """
    def __init__(self, *arg):
        _compoundlongds.__init__(self, *arg) #@UndefinedVariable
        self.ndcls = _compoundlongds

    def copy(self): # override to keep superclass's methods
        return ndarrayCL(self)

    # non-specific code that needs ndcls
    # this code cannot put in ndarray superclass as there is a problem when
    # wrapping methods with same name in superclass
    @ndarraywrapped
    def max(self, axis=None):
        if axis == None:
            return self.ndcls.max(self)
        else:
            return self.ndcls.max(self, axis)

    @ndarraywrapped
    def min(self, axis=None):
        if axis == None:
            return self.ndcls.min(self)
        else:
            return self.ndcls.min(self, axis)

    @ndarraywrapped
    def sum(self, axis=None):
        if axis == None:
            return self.ndcls.sum(self)
        else:
            return self.ndcls.sum(self, axis)

    @ndarraywrapped
    def mean(self, axis=None):
        if axis == None:
            return self.ndcls.mean(self)
        else:
            return self.ndcls.mean(self, axis)

    def sort(self, axis=-1):
        return self.ndcls.sort(self, axis)

    @ndarraywrapped
    def transpose(self, axes=None):
        if axes == None:
            axes = ()
        return self.ndcls.transpose(self, asIterable(axes))

    def put(self, indices, values):
        return self.ndcls.put(self, asIterable(indices), asIterable(values))

    @ndarraywrapped
    def take(self, indices, axis=None):
        return self.ndcls.take(self, asIterable(indices), axis)

    @ndarraywrapped
    def all(self, axis=None):
        if axis == None:
            return self.ndcls.all(self)
        else:
            return self.ndcls.all(self, axis)

    @ndarraywrapped
    def any(self, axis=None):
        if axis == None:
            return self.ndcls.any(self)
        else:
            return self.ndcls.any(self, axis)

    def append(self, other, axis=-1):
        return self.ndcls.append(self, other, axis)

    @ndarraywrapped
    def reshape(self, *shape):
        '''Return a dataset with same data but new shape'''
        if len(shape) == 1:
            shape = asIterable(shape[0])
        return self.ndcls.reshape(self, shape)

class ndarrayCF(ndarray, _compoundfloatds):
    """
    Wrap compound float dataset
    """
    def __init__(self, *arg):
        _compoundfloatds.__init__(self, *arg) #@UndefinedVariable
        self.ndcls = _compoundfloatds

    def copy(self): # override to keep superclass's methods
        return ndarrayCF(self)

    # non-specific code that needs ndcls
    # this code cannot put in ndarray superclass as there is a problem when
    # wrapping methods with same name in superclass
    @ndarraywrapped
    def max(self, axis=None):
        if axis == None:
            return self.ndcls.max(self)
        else:
            return self.ndcls.max(self, axis)

    @ndarraywrapped
    def min(self, axis=None):
        if axis == None:
            return self.ndcls.min(self)
        else:
            return self.ndcls.min(self, axis)

    @ndarraywrapped
    def sum(self, axis=None):
        if axis == None:
            return self.ndcls.sum(self)
        else:
            return self.ndcls.sum(self, axis)

    @ndarraywrapped
    def mean(self, axis=None):
        if axis == None:
            return self.ndcls.mean(self)
        else:
            return self.ndcls.mean(self, axis)

    def sort(self, axis=-1):
        return self.ndcls.sort(self, axis)

    @ndarraywrapped
    def transpose(self, axes=None):
        if axes == None:
            axes = ()
        return self.ndcls.transpose(self, asIterable(axes))

    def put(self, indices, values):
        return self.ndcls.put(self, asIterable(indices), asIterable(values))

    @ndarraywrapped
    def take(self, indices, axis=None):
        return self.ndcls.take(self, asIterable(indices), axis)

    @ndarraywrapped
    def all(self, axis=None):
        if axis == None:
            return self.ndcls.all(self)
        else:
            return self.ndcls.all(self, axis)

    @ndarraywrapped
    def any(self, axis=None):
        if axis == None:
            return self.ndcls.any(self)
        else:
            return self.ndcls.any(self, axis)

    def append(self, other, axis=-1):
        return self.ndcls.append(self, other, axis)

    @ndarraywrapped
    def reshape(self, *shape):
        '''Return a dataset with same data but new shape'''
        if len(shape) == 1:
            shape = asIterable(shape[0])
        return self.ndcls.reshape(self, shape)

class ndarrayCD(ndarray, _compounddoubleds):
    """
    Wrap compound double dataset
    """
    def __init__(self, *arg):
        _compounddoubleds.__init__(self, *arg) #@UndefinedVariable
        self.ndcls = _compounddoubleds

    def copy(self): # override to keep superclass's methods
        return ndarrayCD(self)

    # non-specific code that needs ndcls
    # this code cannot put in ndarray superclass as there is a problem when
    # wrapping methods with same name in superclass
    @ndarraywrapped
    def max(self, axis=None):
        if axis == None:
            return self.ndcls.max(self)
        else:
            return self.ndcls.max(self, axis)

    @ndarraywrapped
    def min(self, axis=None):
        if axis == None:
            return self.ndcls.min(self)
        else:
            return self.ndcls.min(self, axis)

    @ndarraywrapped
    def sum(self, axis=None):
        if axis == None:
            return self.ndcls.sum(self)
        else:
            return self.ndcls.sum(self, axis)

    @ndarraywrapped
    def mean(self, axis=None):
        if axis == None:
            return self.ndcls.mean(self)
        else:
            return self.ndcls.mean(self, axis)

    def sort(self, axis=-1):
        return self.ndcls.sort(self, axis)

    @ndarraywrapped
    def transpose(self, axes=None):
        if axes == None:
            axes = ()
        return self.ndcls.transpose(self, asIterable(axes))

    def put(self, indices, values):
        return self.ndcls.put(self, asIterable(indices), asIterable(values))

    @ndarraywrapped
    def take(self, indices, axis=None):
        return self.ndcls.take(self, asIterable(indices), axis)

    @ndarraywrapped
    def all(self, axis=None):
        if axis == None:
            return self.ndcls.all(self)
        else:
            return self.ndcls.all(self, axis)

    @ndarraywrapped
    def any(self, axis=None):
        if axis == None:
            return self.ndcls.any(self)
        else:
            return self.ndcls.any(self, axis)

    def append(self, other, axis=-1):
        return self.ndcls.append(self, other, axis)

    @ndarraywrapped
    def reshape(self, *shape):
        '''Return a dataset with same data but new shape'''
        if len(shape) == 1:
            shape = asIterable(shape[0])
        return self.ndcls.reshape(self, shape)

class ndarrayRGB(ndarray, _rgbds):
    """
    Wrap RGB dataset
    """
    def __init__(self, *arg):
        _rgbds.__init__(self, *arg) #@UndefinedVariable
        self.ndcls = _rgbds

    def copy(self): # override to keep superclass's methods
        return ndarrayRGB(self)

    def get_red(self, dtype=None):
        if dtype is None:
            dtype = int16
        return self.createRedDataset(dtype)

    def get_green(self, dtype=None):
        if dtype is None:
            dtype = int16
        return self.createGreenDataset(dtype)

    def get_blue(self, dtype=None):
        if dtype is None:
            dtype = int16
        return self.createBlueDataset(dtype)

    def get_grey(self, cweights=None, dtype=None):
        '''Get grey image
        
        Arguments:
        cweights -- optional set of weight for combining the colour channel
        dtype    -- optional dataset type (default is int16)'''
        if dtype is None:
            dtype = int16
        if cweights:
            cweights = asIterable(cweights)
            if len(cweights) != 3:
                raise ValueError, 'three colour channel weights needed'
            csum = float(sum(cweights))
            return self.createGreyDataset(cweights[0]/csum, cweights[1]/csum, cweights[2]/csum, dtype)
        return self.createGreyDataset(dtype)

    red = property(get_red)
    green = property(get_green)
    blue = property(get_blue)
    grey = property(get_grey)

    # non-specific code that needs ndcls
    # this code cannot put in ndarray superclass as there is a problem when
    # wrapping methods with same name in superclass
    @ndarraywrapped
    def max(self, axis=None):
        if axis == None:
            return self.ndcls.max(self)
        else:
            return self.ndcls.max(self, axis)

    @ndarraywrapped
    def min(self, axis=None):
        if axis == None:
            return self.ndcls.min(self)
        else:
            return self.ndcls.min(self, axis)

    @ndarraywrapped
    def sum(self, axis=None):
        if axis == None:
            return self.ndcls.sum(self)
        else:
            return self.ndcls.sum(self, axis)

    @ndarraywrapped
    def mean(self, axis=None):
        if axis == None:
            return self.ndcls.mean(self)
        else:
            return self.ndcls.mean(self, axis)

    def sort(self, axis=-1):
        return self.ndcls.sort(self, axis)

    @ndarraywrapped
    def transpose(self, axes=None):
        if axes == None:
            axes = ()
        return self.ndcls.transpose(self, asIterable(axes))

    def put(self, indices, values):
        return self.ndcls.put(self, asIterable(indices), asIterable(values))

    @ndarraywrapped
    def take(self, indices, axis=None):
        return self.ndcls.take(self, asIterable(indices), axis)

    @ndarraywrapped
    def all(self, axis=None):
        if axis == None:
            return self.ndcls.all(self)
        else:
            return self.ndcls.all(self, axis)

    @ndarraywrapped
    def any(self, axis=None):
        if axis == None:
            return self.ndcls.any(self)
        else:
            return self.ndcls.any(self, axis)

    def append(self, other, axis=-1):
        return self.ndcls.append(self, other, axis)

    @ndarraywrapped
    def reshape(self, *shape):
        '''Return a dataset with same data but new shape'''
        if len(shape) == 1:
            shape = asIterable(shape[0])
        return self.ndcls.reshape(self, shape)

# dictionary to map from dtype to nd array class
__dtype2jythoncls = { bool:ndarrayA, int8:ndarrayB, int16:ndarrayS, int32:ndarrayI, int64:ndarrayL,
                     float32:ndarrayF, float64:ndarrayD,
                     complex64:ndarrayC, complex128:ndarrayZ }

__cdtype2jythoncls = { cint8:ndarrayCB, cint16:ndarrayCS, cint32:ndarrayCI, cint64:ndarrayCL,
                     cfloat32:ndarrayCF, cfloat64:ndarrayCD,
                     complex64:ndarrayC, complex128:ndarrayZ }

__dtype2javacls = { bool:_booleands, int8:_byteds, int16:_shortds, int32:_integerds, int64:_longds,
                   float32:_floatds, float64:_doubleds,
                   complex64:_complexfloatds, complex128:_complexdoubleds }

__cdtype2javacls = { cint8:_compoundbyteds, cint16:_compoundshortds, cint32:_compoundintegerds, cint64:_compoundlongds,
                   cfloat32:_compoundfloatds, cfloat64:_compounddoubleds }

@ndarraywrapped
def arange(start, stop=None, step=1, dtype=None):
    '''Create a 1D dataset of given type where values range from specified start up to
    but not including stop in given steps

    Arguments:
    start -- optional starting value, defaults to 0
    stop  -- exclusive stop value
    step  -- difference between neighbouring values, defaults to 1
    dtype -- defaults to None which means the type is inferred from given start, stop, step values
    '''
    if stop == None:
        stop = start
        start = 0
    if dtype == None:
        if type(start) is _types.ComplexType or type(stop) is _types.ComplexType or type(step) is _types.ComplexType: 
            dtype = complex128
        elif type(start) is _types.FloatType or type(stop) is _types.FloatType or type(step) is _types.FloatType: 
            dtype = float64
        elif type(start) is _types.IntType or type(stop) is _types.IntType or type(step) is _types.IntType: 
            dtype = int32
        else:
            raise ValueError, "Unknown or invalid type of input value"
    if dtype == bool:
        return None

    return __dtype2jythoncls[dtype].arange(start, stop, step)

@ndarraywrapped
def array(object, dtype=None):
    '''Create a dataset of given type from a sequence or JAMA matrix'''
    if not isinstance(object, list):
        if isinstance(object, _matrix): # cope with JAMA matrices
            if dtype == None:
                dtype = float64
            object = object.getArray()

    if dtype == None:
        dtype = _abstractds.getDTypeFromObject(object) #@UndefinedVariable

    return __dtype2jythoncls[dtype].createFromObject(object)

@ndarraywrapped
def ones(shape, dtype=float64, elements=1):
    '''Create a dataset filled with 1'''
    if elements != 1:
        return __cdtype2jythoncls[dtype].ones(elements, asIterable(shape))
    return __dtype2jythoncls[dtype].ones(asIterable(shape))

def zeros(shape, dtype=float64, elements=1):
    '''Create a dataset filled with 0'''
    if elements != 1:
        return __cdtype2jythoncls[dtype](elements, asIterable(shape))
    return __dtype2jythoncls[dtype](asIterable(shape))

empty = zeros

@ndarraywrapped
def zeros_like(a):
    return _abstractds.zeros(a)

empty_like = zeros_like

@ndarraywrapped
def ones_like(a):
    return _abstractds.zeros(a).fill(1)

def linspace(start, stop, num=50, endpoint=True, retstep=False):
    '''Create a 1D dataset from start to stop in given number of steps
    
    Arguments:
    start    -- starting value
    stop     -- stopping value
    num      -- number of steps, defaults to 50
    endpoint -- if True (default), include the stop value
    retstep  -- if False (default), do not include the calculated step value as part of return tuple
    '''
    if not endpoint:
        step = (stop - start) / (num - 1.)
        stop -= step

    dtype = _abstractds.getDTypeFromObject((start, stop))
    if dtype < float64:
        dtype = float64

    if dtype >= complex64:
        dtype = complex128

        if type(start) is _types.IntType:
            start = start+0j
        if type(stop) is _types.IntType:
            stop = stop+0j
        rresult = _dsutils.linSpace(start.real, stop.real, num, float64)
        iresult = _dsutils.linSpace(start.imag, stop.imag, num, float64)
        result = Sciwrap(_complexdoubleds(rresult, iresult))
        del rresult, iresult
    else:
        result = Sciwrap(_dsutils.linSpace(start, stop, num, dtype))

    if retstep:
        step = result[1] - result[0]
        return (step, result)
    else:
        return result

@ndarraywrapped
def _logspace(start, stop, num=50, endpoint=True, base=10.0):
    '''Create a 1D dataset of values equally spaced on a logarithmic scale'''
    result = linspace(start, stop, num, endpoint)
    return _maths.power(base, result)

@ndarraywrapped
def logspace(start, stop, num=50, endpoint=True, base=10.0):
    '''Create a 1D dataset of values equally spaced on a logarithmic scale'''
    if _pycomplex(start).imag == 0 and _pycomplex(stop).imag == 0:
        return _dsutils.logSpace(start, stop, num, base, endpoint)
    else:
        result = linspace(start, stop, num, endpoint)
        return _maths.power(base, result)

@ndarraywrapped
def eye(N, M=None, k=0, dtype=float64):
    if M == None:
        M = N
    return _dsutils.eye(N, M, k, dtype)

def identity(n, dtype=float64):
    return eye(n,n,0,dtype)

@ndarraywrapped
def diag(v, k=0):
    x = asDataset(v)
    return _dsutils.diag(x, k)

@ndarraywrapped
def diagflat(v, k=0):
    x = asDataset(v).flatten()
    return _dsutils.diag(x, k)

def take(a, indices, axis=None):
    return a.take(indices, axis)

@ndarraywrapped
def put(a, indices, values):
    return a.put(indices, values)

@ndarraywrapped
def concatenate(a, axis=0):
    return _dsutils.concatenate(toList(a), axis)

@ndarraywrapped
def vstack(tup):
    return _dsutils.concatenate(toList(tup), 0)

@ndarraywrapped
def hstack(tup):
    return _dsutils.concatenate(toList(tup), 1)

@ndarraywrapped
def dstack(tup):
    return _dsutils.concatenate(toList(tup), 2)

@ndarraywrapped
def split(ary, indices_or_sections, axis=0):
    return _dsutils.split(ary, indices_or_sections, axis, True)

@ndarraywrapped
def array_split(ary, indices_or_sections, axis=0):
    return _dsutils.split(ary, indices_or_sections, axis, False)

def vsplit(ary, indices_or_sections):
    return split(ary, indices_or_sections, 0)

def hsplit(ary, indices_or_sections):
    return split(ary, indices_or_sections, 1)

def dsplit(ary, indices_or_sections):
    return split(ary, indices_or_sections, 2)

@ndarraywrapped
def sort(a, axis=-1):
    return _dsutils.sort(a, axis)

@ndarraywrapped
def tile(a, reps):
    return _dsutils.tile(a, asIterable(reps))

@ndarraywrapped
def repeat(a, repeats, axis=-1):
    return _dsutils.repeat(a, asIterable(repeats), axis)

@ndarraywrapped
def cast(a, dtype):
    return _dsutils.cast(a, dtype)

@ndarraywrapped
def any(a, axis=None):
    if axis == None or (a.rank == 1 and axis == 0):
        return a.any()
    return a.any(axis)

@ndarraywrapped
def all(a, axis=None):
    if axis == None or (a.rank == 1 and axis == 0):
        return a.all()
    return a.all(axis)

def squeeze(a):
    a.squeeze()
    return a

@ndarraywrapped
def transpose(a, axes=None):
    if axes == None:
        axes = ()
    return _dsutils.transpose(a, asIterable(axes))

@ndarraywrapped
def swapaxes(a, axis1, axis2):
    return _dsutils.swapAxes(a, axis1, axis2)

def argmax(a, axis=None):
    return a.argmax(axis)

def argmin(a, axis=None):
    return a.argmin(axis)

@ndarraywrapped
def maximum(a, b):
    return _dsutils.maximum(a, b)

@ndarraywrapped
def minimum(a, b):
    return _dsutils.minimum(a, b)

def meshgrid(*a):
    axes = [ asDataset(x) for x in reversed(a) ]
    coords = _dsutils.meshGrid(axes)
    return tuple([ Sciwrap(x) for x in reversed(coords) ])

@ndarraywrapped
def indices(dimensions, dtype=int32):
    ind = _dsutils.indices(asIterable(dimensions))
    if dtype != int32:
        ind = _dsutils.cast(ind, dtype)
    return ind

@ndarraywrapped
def norm(a, allelements=True):
    '''Normalise array so all elements lie between 0 and 1
    Keyword argument:
    allelements -- if True, then normalise for all elements rather than per-element
    '''
    if isinstance(a, _abscompoundds):
        return _dsutils.norm(a, allelements)
    return _dsutils.norm(a)

@ndarraywrapped
def compoundarray(a, view=True):
    '''Create a compound array from an nd array by grouping last axis items into compound items
    '''
    return _dsutils.createCompoundDatasetFromLastAxis(a, view)

