#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
#    Project: Fast Azimuthal integration
#             https://github.com/kif/pyFAI
#
#
#    Copyright (C) European Synchrotron Radiation Facility, Grenoble, France
#
#    Authors: Jérôme Kieffer <Jerome.Kieffer@ESRF.eu>
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
"""
pyFAI-average is a small utility that averages out a serie of files,
for example for dark, or flat or calibration images
"""
__author__ = "Jerome Kieffer, Picca Frédéric-Emmanuel"
__contact__ = "Jerome.Kieffer@ESRF.eu"
__license__ = "GPLv3+"
__copyright__ = "European Synchrotron Radiation Facility, Grenoble, France"
__date__ = "03/05/2013"
__status__ = "development"

import os
import sys
import fabio
import logging
import pyFAI, pyFAI.utils
from optparse import OptionParser
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger("average")

def main():
    usage = "usage: %prog [options] [options] -o output.edf file1.edf file2.edf ..."
    version = "%prog " + pyFAI.version
    description = """
    This tool can be used to average out a set of dark current images using
    mean or median filter (along the image stack). One can also reject outliers
    be specifying a cutoff (remove cosmic rays / zingers from dark)
    """
    epilog = """It can also be used to merge many images from the same sample when using a small beam
    and reduce the spotty-ness of Debye-Sherrer rings. In this case the "max-filter" is usually
    recommended.
    """
    parser = OptionParser(usage=usage, version=version,
                          description=description, epilog=epilog)
    parser.add_option("-o", "--output", dest="output",
                      type='string', default=None,
                      help="Output/ destination of average image")
    parser.add_option("-m","--method", dest="method",
                      type='string', default="mean",
                      help="Method used for averaging, can be 'mean'(default) or 'median', 'min' or 'max'")
    parser.add_option("-c", "--cutoff", dest="cutoff", type="float",default=None,
                  help="Take the mean of the average +/- cutoff * std_dev.")
    parser.add_option("-f", "--format", dest="format", type="string", default="edf",
                  help="Output file/image format (by default EDF)")
    parser.add_option("-v", "--verbose", action="store_true", dest="verbose", default=False,
                      help="switch to verbose/debug mode")
    (options, args) = parser.parse_args()
    if options.verbose:
        pyFAI.utils.logger.setLevel(logging.DEBUG)
    else:
        pyFAI.utils.logger.setLevel(logging.INFO)

    images = pyFAI.utils.expand_args(args)
    if  options.output:
        output = options.output
    else:
        output = "%s_%s" % (os.path.commonprefix(args), options.method)
        if options.cutoff:
            output += "_cutoff_%s_std" % options.cutoff
        output += "_%s_frames.%s" % (len(args), options.format)
    if images:
        dataout = pyFAI.utils.averageImages(images, filter_=options.method, cutoff=options.cutoff,
                                            threshold=0, format=options.format, output=output)
if __name__ == "__main__":
    main()
